#include "term.h"
#include "wam.h" // access hs addresses
#include "global.h"
#include "tokenizer.h"
#include <memory>

namespace lp {

	node_const_iterator& node_const_iterator::operator++()
	{
		auto update_parent = [&](const term* curr, const term* parent){
			auto at = pars.find(curr);
			if (at == pars.end()) {
				pars.insert(std::make_pair(curr,parent));
			} else {
				at->second = parent;
			}
		};

		if (nptr->is_function()) {
			// Move down to left
			update_parent(nptr+1,nptr);
			++nptr;
		} else if (nptr->is_variable() && nptr != nptr->get_ptr()) {
			auto p = vars.insert(nptr);
			if (!p.second) {
				if (accept_cycle) {
					goto move_up;
				} else {
					throw cyclic_term(&*nptr);
				}
			}
			update_parent(nptr->get_ptr(),nptr);
			nptr = nptr->get_ptr();
		} else {
			// Leaf node: move up and then down next sibling
move_up:
			for (;;) {
				auto at = pars.find(nptr);
				if (at == pars.end()) {
					nptr = nullptr;
					break; // finished
				}
				const term* p = at->second;
				if (p->is_variable() && p != p->get_ptr()) {
					vars.erase(p);
					pars.erase(nptr);
					nptr = p; // continue up
				} else {
					assert(p->is_function());
					const int nptr_k = (nptr - p);
					if (nptr_k < p->arity()) {
						// Move down here
						nptr = p+nptr_k+1;
						update_parent(nptr,p);
						break;
					} else {
						// k == ar, so continue up
						// this subtree is finished, so clear pars
						pars.erase(nptr);
						nptr = p;
					}
				} // is function
			} // while moving up
		} // if leaf
		return *this;
	}


	term* deref_nocycle(term* t, std::set<const term*>& vars)
	{
		while (t->is_variable() && t->get_ptr() != t) {
			auto p = vars.insert(t);
			if (!p.second) return nullptr;
			t = t->get_ptr();
		}
		return t;
	}

	const term* is_cyclic(const term* t)
	{
		try {
			for (auto i = t->node_begin() ; i != t->node_end(); ++i) { }
		} catch (const cyclic_term& ct) {
			assert(ct.ptr);
			return ct.ptr;
		}
		return nullptr;
	}

	void print_infix(
		std::ostream& os, 
		const term* t, 
		std::map<const term*,std::string>& vname)
	{
		std::set<const term*> vars;
		std::set<const term*> singletons;
		if (pretty_print.is_int() && pretty_print.get_int() >= 2) {
			std::map<const term*,int> vcount;
			// Determine which variables only occur once
			std::for_each(t->begin(true),t->end(true),[&](const term& n){ if (n.is_variable()) ++vcount[&n]; });
			for (auto& p : vcount) {
				if (p.second == 1) singletons.insert(p.first);
			}
		}

		auto print_seq = [&](const term* n){
			n = deref_nocycle(n,vars);
			if (!n) {
				os << "...";
				return;
			} else if (!n->is_function(sequence_id,2)) {
				// Not sequence, print as normal
				print_infix_r(os,n,vname,singletons,vars);
				return;
			} else {
				print_infix_r(os,n+1,vname,singletons,vars); // first in seq
				for (n = deref_nocycle(n+2,vars) ; n && n->is_function(sequence_id,2); n = deref_nocycle(n+2,vars)) {
					os << ", ";
					print_infix_r(os,n+1,vname,singletons,vars);
				}
				if (!n) {
					os << "...";
					return;
				}
				os << ", ";
				print_infix_r(os,n,vname,singletons,vars); // last elem in seq
			}
		};

		t = deref_nocycle(t,vars);
		if (!t) {
			os << "...";
		} else if (t->is_constant(if_id)) {
			os << ":-";
		} else if (t->get_type() == if_id) {
			switch (t->arity()) {
			case 1:
				os << ":- ";
				print_seq(t+1);				
				break;
			case 2:
				print_infix_r(os,t+1,vname,singletons,vars);
				os << " :- ";
				print_seq(t+2);
				break;
			default:
				print_infix_r(os,t,vname,singletons,vars);
				break;
			}
		} else {
			print_infix_r(os,t,vname,singletons,vars);
		}
	}

	inline std::string int2str(int k)
	{
		assert(k >= 0);
		if (k < 26) return std::string(1,'A' + k);
		else return std::string("_G") + std::to_string(long long(k-26));
	}


	void print_infix_r(
		std::ostream& os, 
		const term* t,
		std::map<const term*,std::string>& vname,
		std::set<const term*>& singletons,
		std::set<const term*>& vars)
	{
		switch (t->get_type()) {
		case term::VAR: case term::EVAR:
			{
				const term* d = deref_nocycle(t,vars);
				if (!d) {
					os << "...";
				} else if (d->is_variable()) {
					// Make/Print index
					if (singletons.find(d) != singletons.end()) {
						assert(vname.find(d) == vname.end());
						os << "_";
					} else {
						auto at = vname.find(d);
						if (at == vname.end()) {
							const std::string s = int2str(vname.size());
							os << s;
							vname.insert(std::make_pair(d,std::move(s)));
						} else {
							os << at->second;
						}
					}
				} else {
					return print_infix_r(os,d,vname,singletons,vars);
				}
				break;
			}
		case term::CON:
			os << fmap.get_data(t->get_const());
			break;
		case term::INT:
			os << t->get_int();
			break;
		case term::FLT:
			{
				// Always print at least one decimal
				double intpart;
				if (std::modf(t->get_double(),&intpart) == 0) {
					// Is integer, print .0
					const auto prec = os.precision();
					os << std::fixed << std::setprecision(1) << t->get_double() 
						<< std::setprecision(prec) << std::defaultfloat;
				} else {
					os << t->get_double();
				}
			}
			break;
		case 0: 
			//assert(false); 
			os << "NIL";
			break;
		case term::LIS:
			{
				// Print list
				os << "[";
				const term* ptr = t;
				for (;;) {
					print_infix_r(os, ptr + 1, vname,singletons,vars);
					ptr = deref_nocycle(ptr+2,vars);
					if (!ptr) {
						os << "...";
						break;
					} else if (!ptr->is_list()) {
						// Print last element
						if (!ptr->is_constant(empty_list_id)) {
							os << "|";
							print_infix_r(os, ptr, vname,singletons,vars);
						}
						os << "]";
						break;
					} // else: continue
					os << ",";
				}
			}
			break;
		default: // functor
			assert(t->is_function() && t->arity() >= 0);
			assert( fmap.get_data(t->get_fun()).is_atom() );
			// Check if operator
			auto precedence = [&](const term* n) -> int {
				if (n->is_function()) {
					switch (n->arity()) {
					case 1:
						{
							const auto& dat = fmap.get_data(n->get_fun());
							assert( dat.is_atom() );
							try {
								return prolog_dic.precedence(dat.get_atom(), Token::xf | Token::yf | Token::fx | Token::fy);
							} catch (dic_not_found) { }
						}
						break;
					case 2:
						{
							const auto& dat = fmap.get_data(n->get_fun());
							assert( dat.is_atom() );
							try {
								return prolog_dic.precedence(dat.get_atom(), Token::xfy | Token::yfx | Token::xfx);
							} catch (dic_not_found) { }
						}
						break;
					}
				} // else:
				return 0;
			};
			switch (t->arity()) {
			case 1:
				{
					const auto& dat = fmap.get_data(t->get_fun());
					assert( dat.is_atom() );
					const std::string s = dat.get_atom();
					if (prolog_dic.has_prefix_op(s)) {
						const int p = precedence(t);
						bool par = (p <= precedence(t+1));
						os << s;
						if (std::isalnum(s.back())) os << " ";
						if (par) os << "(";
						print_infix_r(os,t+1,vname,singletons,vars);
						if (par) os << ")";
					} else if (prolog_dic.has_suffix_op(s)) {
						const int p = precedence(t);
						bool par = (p <= precedence(t+1));
						if (par) os << "(";
						print_infix_r(os,t+1,vname,singletons,vars);
						if (par) os << ")";
						if (std::isalnum(s.front())) os << " ";
						os << s;
					} else {
						os << s << "(";
						print_infix_r(os,t+1,vname,singletons,vars);
						os << ")";
					}
				}
				break;
			case 2:
				{
					const auto& dat = fmap.get_data(t->get_fun());
					const std::string s = dat.get_atom();
					if (prolog_dic.has_infix_op(s)) {
						const int p = precedence(t);
						bool par = (p <= precedence(t+1));
						if (par) os << "(";
						print_infix_r(os,t+1,vname,singletons,vars);
						if (par) os << ")";
						if (std::isalnum(s.front())) os << " ";
						os << s;
						if (std::isalnum(s.back())) os << " ";
						par = (p <= precedence(t+2));
						if (par) os << "(";
						print_infix_r(os,t+2,vname,singletons,vars);
						if (par) os << ")";
					} else {
						os << s << "(";
						print_infix_r(os,t+1,vname,singletons,vars);
						os << ",";
						print_infix_r(os,t+2,vname,singletons,vars);
						os << ")";
					}
				}
				break;
			default:
				{
					os << fmap.get_data(t->get_fun());
					if (t->arity() > 0) {
						os << "(";
						print_infix_r(os, t + 1, vname,singletons,vars);
						for (int k = 2; k <= t->arity(); ++k) {
							os << ",";
							print_infix_r(os, t + k, vname,singletons,vars);
						}
						os << ")";
					}
				}
			} // end inner switch (prefix/infix/suffix)
		} // end outer switch (CON/INT/FLT/...)
	}


	void term::print(std::ostream& os) const
	{
		switch (type) {
		case 0: os << "NIL " << arity(); break;
		case EVAR: os << "EVAR " << get_ptr(); break;
		case VAR: os << "VAR " << get_ptr(); break;
		case CON: os << "CON "; os << fmap.get_data(get_const()); break;
		case INT: os << "INT " << get_int(); break;
		case FLT: os << "FLT " << get_double(); break;
		default:
			os << lp::fmap.get_data(get_type()) << "/" << arity();
		}
	}

	void print_all(std::ostream& os, const term* t)
	{
		os << t << ": "; t->print(os); os << "\n";
		if (t->is_variable() && t != t->get_ptr()) {
			print_all(os,t->get_ptr());
		} else if (t->is_function()) {
			for (int k = 1; k <= t->arity(); ++k) {
				os << " " << (t+k) << ": ";
				(t+k)->print(os);
				os << "\n";
			}
			for (int k = 1; k <= t->arity(); ++k) {
				if ((t+k)->is_variable() && (t+k) != (t+k)->get_ptr()) {
					print_all(os,(t+k)->get_ptr());
				}
			}
		} else {
			os <<  " " << t << ": "; t->print(os); os << "\n";
		}
	}


	int rcompare(const term* t1, const term* t2)
	{
		std::stack<const term*> pdl;
		pdl.push(t2);
		pdl.push(t1);
		const int MAX_RECURSE = 10000; // TODO: make adjustable parameter (same as MAX_UNIFICATION)
		int i = 0;
		for ( ; !pdl.empty() && i < MAX_RECURSE; ++i) {
			const term* x = deref(deref(pdl.top()));
			pdl.pop();
			const term* y = deref(deref(pdl.top()));
			pdl.pop();

			// variables < float < int < atom < compound
			if (x->is_list()) {
				if (!y->is_list()) return 1;
				pdl.push(y+2);
				pdl.push(x+2);
				pdl.push(y+1);
				pdl.push(x+1);
			} else if (y->is_list()) {
				return -1;
			} else if (x->is_function()) {
				if (!y->is_function()) return 1;
				if (x->arity() < y->arity()) return -1;
				if (x->arity() > y->arity()) return 1;
				int r = std::strcmp(
					lp::fmap.get_data(x->get_type()).get_atom(),
					lp::fmap.get_data(y->get_type()).get_atom());
				if (r != 0) return r;
				for (int k = x->arity(); k > 0; --k) {
					pdl.push(y+k);
					pdl.push(x+k);
				}
			} else if (y->is_function()) {
				assert(!x->is_function());
				return -1;
			} else {
				// CON, INT, or FLT
				assert(x->is_constant() || x->is_int() || x->is_double());
				assert(y->is_constant() || y->is_int() || y->is_double());
				if (x->get_type() < y->get_type()) return -1;
				if (x->get_type() > y->get_type()) return 1;
				// Same symbol
				switch (x->get_type()) {
				case term::EVAR: case term::VAR:
					// No renaming: Compare addresses
					if (x < y) return -1;
					if (x > y) return 1;
					break; // else: continue
				case term::CON: {
					const int r = std::strcmp(fmap.get_data(x->arity()).get_atom(),fmap.get_data(y->arity()).get_atom());
					if (r != 0) return r;
					break;
								}
				case term::FLT: {
					const double diff = x->get_double() - y->get_double();
					if (diff < 0) return -1;
					if (diff > 0) return 1;
					break;
								}
				case term::INT: {
					const long long diff = x->get_int() - y->get_int();
					if (diff < 0) return -1;
					if (diff > 0) return 1;
					break;
								}
				default:
					std::cerr << "Error: unrecognized term of type " << x->get_type() << "\n";
					assert(false);
					std::exit(1);
				}
			}
		} // for
		if (i >= MAX_RECURSE) {
			DEBUG_WARNING(std::cerr << "Warning: rcompare reached maximum recursion " << MAX_RECURSE << " (probably cyclic term)\n";);
			throw cyclic_term();
			//if (t1 < t2) return -1;
			//if (t1 > t2) return 1;
			// else: return 0
		}
		return 0;
	}

	bool is_variant(
		const term* t1, 
		const term* t2, 
		std::map<const term*,const term*>& vm1,
		std::map<const term*,const term*>& vm2)
	{
		// Compare symbols
		std::stack<const term*> pdl;
		pdl.push(t2);
		pdl.push(t1);
		const int MAX_RECURSE = 10000; // TODO: make adjustable parameter (same as MAX_UNIFICATION)
		int i = 0;
		for ( ; !pdl.empty() && i < MAX_RECURSE; ++i) {
			const term* x = deref(pdl.top());
			pdl.pop();
			const term* y = deref(pdl.top());
			pdl.pop();
			// variables < float < int < atom < compound
			if (x->is_list()) {
				if (!y->is_list()) return false;
				pdl.push(y+2);
				pdl.push(x+2);
				pdl.push(y+1);
				pdl.push(x+1);
			} else if (y->is_list()) {
				return false;
			} else if (x->is_function()) {
				if (!y->is_function()) return false;
				if (x->arity() != y->arity()) return false;
				int r = std::strcmp(
					lp::fmap.get_data(x->get_type()).get_atom(),
					lp::fmap.get_data(y->get_type()).get_atom());
				if (r != 0) return false;
				for (int k = x->arity(); k > 0; --k) {
					pdl.push(y+k);
					pdl.push(x+k);
				}
			} else if (y->is_function()) {
				assert(!x->is_function());
				return false;
			} else {
				switch (x->get_type()) {
				case term::EVAR: case term::VAR:
					{
						if (!y->is_variable()) return false;
						// Allow renaming
						auto at1 = vm1.find(x);
						auto at2 = vm2.find(y);
						if (at1 == vm1.end()) {
							// New variable in s
							if (at2 == vm2.end()) {
								// Also new in t, make new binding
								vm1.insert(std::make_pair(x,y));
								vm2.insert(std::make_pair(y,x));
							} else {
								// t already binds this one
								assert(x != y);
								return false;
							}
						} else {
							// else: at1 is bound to previous
							if (at2 == vm2.end()) {
								// s is bound, but not t
								assert(x != y);
								return false;
							} else {
								// Both are bound
								if (at1->second != at2->first || at2->second != at1->first) {
									assert(x != y);
									return false;
								}
							}
						}
					}
					break;
				case term::CON: {
					if (!y->is_constant()) return false;
					if (std::strcmp(lp::fmap.get_data(x->arity()).get_atom(),lp::fmap.get_data(y->arity()).get_atom()) != 0) {
						return false;
					}
					break;
								}
				case term::FLT:
					{
						if (!y->is_double()) return false;
						if (x->get_double() != y->get_double()) return false;
					}
					break;
				case term::INT:
					{
						if (!y->is_int()) return false;
						if (x->get_int() != y->get_int()) return false;
					}
					break;
				default:
					std::cerr << "Error: unrecognized term of type " << x->get_type() << "\n";
					assert(false);
					std::exit(1);
				}
			}
		} // for

		if (i >= MAX_RECURSE) {
			DEBUG_WARNING(std::cerr << "Warning: is_variant reached recursion limit of " << MAX_RECURSE << " (probably cyclic term)\n");
			throw cyclic_term();
			//return false;
		}
		return true;
	}


	term eval(const term* t, int mr)
	{
		if (mr <= 0) throw useful::non_numeric(); // TODO: throw cyclic_term()?
		t = deref(t);
		switch (t->get_type()) {
		case term::INT:
		case term::FLT:
			return *t;
		case lp::plus_id:
			if (t->arity() == 2) {
				term a1 = eval(t+1,mr-1);
				term a2 = eval(t+2,mr-1);
				if (a1.is_int()) {
					if (a2.is_int()) return term(a1.get_int() + a2.get_int());
					else return term(a1.get_int() + a2.get_double());
				} else {
					if (a2.is_int()) return term(a1.get_double() + a2.get_int());
					else return term(a1.get_double() + a2.get_double());
				}
			}
			break;
		case lp::minus_id:
			if (t->arity() == 1) {
				term a1 = eval(t+1,mr-1);
				if (a1.is_int()) return term(-a1.get_int());
				else return term(-a1.get_double());
			} else if (t->arity() == 2) {
				term a1 = eval(t+1,mr-1);
				term a2 = eval(t+2,mr-1);
				if (a1.is_int()) {
					if (a2.is_int()) return term(a1.get_int() - a2.get_int());
					else return term(a1.get_int() - a2.get_double());
				} else {
					if (a2.is_int()) return term(a1.get_double() - a2.get_int());
					else return term(a1.get_double() - a2.get_double());
				}
			}
			break;
		case lp::mul_id:
			if (t->arity() == 2) {
				term a1 = eval(t+1,mr-1);
				term a2 = eval(t+2,mr-1);
				if (a1.is_int()) {
					if (a2.is_int()) return term(a1.get_int() * a2.get_int());
					else return term(a1.get_int() * a2.get_double());
				} else {
					if (a2.is_int()) return term(a1.get_double() * a2.get_int());
					else return term(a1.get_double() * a2.get_double());
				}
			}
			break;
		case lp::div_id:
			if (t->arity() == 2) {
				term a2 = eval(t+2,mr-1);
				if (a2.is_int() && a2.get_int() == 0 || a2.is_double() && a2.get_double() == 0.0) {
					throw non_numeric();
				}
				term a1 = eval(t+1,mr-1);
				if (a1.is_int()) {
					if (a2.is_int()) {
						const double r = double(a1.get_int()) / a2.get_int();
						if (r == long long(r)) return term(long long(r));
						else return term(r);
					} else return term(a1.get_int() / a2.get_double());
				} else {
					if (a2.is_int()) return term(a1.get_double() / a2.get_int());
					else return term(a1.get_double() / a2.get_double());
				}
			}
			break;
		case lp::star2_id:
		case lp::pow_id:
			if (t->arity() == 2) {
				term a1 = eval(t+1,mr-1);
				term a2 = eval(t+2,mr-1);
				if (a1.is_int()) {
					if (a2.is_int()) {
						const double r = std::pow(double(a1.get_int()), double(a2.get_int()));
						if (r == long long(r)) return term(long long(r));
						else return term(r);
					} else return term(std::pow(double(a1.get_int()), a2.get_double()));
				} else {
					if (a2.is_int()) return term(std::pow(a1.get_double(), double(a2.get_int())));
					else return term(std::pow(a1.get_double(), a2.get_double()));
				}
			}
			break;
		default:
			break;
		}
		throw useful::non_numeric();
	}

	int num_compare(const term* x, const term* y)
	{
		const term a = eval(x);
		const term b = eval(y);
		if (a.is_int()) {
			if (b.is_int()) {
				return a.get_int() < b.get_int() ? -1 : (a.get_int() > b.get_int() ? 1 : 0);
			} else {
				return a.get_int() < b.get_double() ? -1 : (a.get_int() > b.get_double() ? 1 : 0);
			}
		} else {
			if (b.is_int()) {
				return a.get_double() < b.get_int() ? -1 : (a.get_double() > b.get_int() ? 1 : 0);
			} else {
				return a.get_double() < b.get_double() ? -1 : (a.get_double() > b.get_double() ? 1 : 0);
			}
		}
	}

	term* term::head()
	{
		term* t = deref(this);
		if (t->get_type() == lp::if_id) {
			switch (t->arity()) {
			case 1:
				return nullptr;
			case 2: return deref(t+1);
			}
		} else if (t->is_constant(if_id)) {
			return nullptr;
		}
		return t;
	}

	term* term::body()
	{
		term* t = deref(this);
		if (t->get_type() == lp::if_id) {
			switch (t->arity()) {
			case 1: return deref(t+1);
			case 2: return deref(t+2);
			}
		}
		return nullptr;
	}

} // namespace lp


#ifndef NDEBUG
#include <set>
namespace lp {
	namespace {
		int newc = 0;
		int delc = 0;
		int del0 = 0;
		int newa = 0;
		int dela = 0;
		int dela0 = 0;
		std::set<const term*> allocs; // current allocations
		std::set<const term*> alloc_history; // all allocations ever made

		std::set<const term*> allocs_array_head; // current array allocations
		std::set<const term*> allocs_array_all; // current array allocations
		std::set<const term*> alloc_array_history; // things allocated indirectly with []
	}

	void* term::operator new(std::size_t s)
	{
		++newc;
		term* ptr = static_cast<term*>(::operator new(s)); // "delegate" to global operator new
		//std::cerr << "====NEW: " << ptr << " =========\n";
		assert( allocs.find(ptr) == allocs.end() );
		allocs.insert(ptr);
		alloc_history.insert(ptr);
		return ptr;
	}

	void* term::operator new[](std::size_t s)
	{
		++newa;
		term* ptr = static_cast<term*>(::operator new[](s)); // "delegate" to global operator new[]
		//std::cerr << "====NEW[]: " << ptr << " =========\n";
		assert( allocs.find(ptr) == allocs.end() );
		allocs_array_head.insert(ptr);
		allocs_array_all.insert(ptr);
		alloc_array_history.insert(ptr);
		for (term* p = ptr+1; p != ptr+s; ++p) {
			allocs_array_all.insert(ptr);
			alloc_array_history.insert(p);
		}
		return ptr;
	}

	void term::operator delete(void* p)
	{
		if (!p) {
			++del0;
			return;
		}
		++delc;
		const term* t = static_cast<const term*>(p);
		auto at = allocs.find(t);
		if (at == allocs.end()) {
			if (alloc_history.find(t) == alloc_history.end()) {
				// Never allocated in the first place
				if (alloc_array_history.find(t) == alloc_array_history.end()) {
					std::cerr << "ERROR: trying to delete never allocated: " << t << "\n";
					if (alloc_array_history.find(t) != alloc_array_history.end()) {
						if (allocs_array_all.find(t) == allocs_array_all.end()) {
							std::cerr << "ERROR: it was allocated as array\n";
						} else {
							std::cerr << "ERROR: it IS allocated as array, head? " 
								<< (allocs_array_head.find(t) != allocs_array_head.end()) << "\n";
						}
					}
				} else {
					std::cerr << "ERROR: trying to delete already deleted: " << t << "\n";
				}
			} else {
				std::cerr << "ERROR: trying to delete already deleted: " << t << "\n";
			}
			assert(false);
			exit(1);
		}
		allocs.erase(at);
		//std::cerr << "====DEL: " << t << " =========\n";
		::operator delete(p); // "delegate" to global operator delete
	}

	void term::operator delete[](void* p)
	{
		if (!p) {
			++dela0;
			return;
		}
		++dela;
		const term* t = static_cast<const term*>(p);
		auto at = allocs_array_head.find(t);
		if (at == allocs_array_head.end()) {
			std::cerr << "ERROR: trying to delete[]: " << t << "\n";
			assert(false);
			exit(1);
		}
		allocs_array_head.erase(at);
		//std::cerr << "====DEL[]: " << t << " =========\n";
		::operator delete(p); // "delegate" to global operator delete
	}


	void check_memory()
	{
		cerr << "terms allocated:     " << newc << "\n";
		cerr << "terms deallocated: " << delc << " \n";
		cerr << "terms null: " << del0 << " \n";
		cerr << "terms[] allocated:     " << newa << "\n";
		cerr << "terms[] deallocated: " << dela << " \n";
		cerr << "terms[] null: " << dela0 << " \n";

		if (!allocs.empty()) {
			cerr << "Forgot to deallocate " << allocs.size() << " objects\n";
			for_each(allocs.begin(),allocs.end(),[&](const term* a){
				if (a->is_variable()) {
					cerr << "Unallocated variable " << a << " => " << a->get_ptr() << "\n";
				} else {
					cerr << "Unallocated (" << a->get_type() << ") " << *a << "\n";
				}
			});
			assert(false);
			exit(1);
		}
	}
}
#endif

